/*
 * Decompiled with CFR 0.152.
 */
package com.sts15.enderdrives.db;

import appeng.api.stacks.AEItemKey;
import com.sts15.enderdrives.config.serverConfig;
import com.sts15.enderdrives.db.StoredEntry;
import com.sts15.enderdrives.db.TapeKey;
import com.sts15.enderdrives.items.TapeDiskItem;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.zip.CRC32;
import net.minecraft.core.HolderLookup;
import net.minecraft.core.RegistryAccess;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.Tag;
import net.minecraft.nbt.TagParser;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.storage.LevelResource;
import net.neoforged.neoforge.server.ServerLifecycleHooks;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class TapeDBManager {
    private static final Logger LOGGER = LogManager.getLogger((String)"EnderDrives-TapeDB");
    private static final Map<UUID, TapeDriveCache> activeCaches = new ConcurrentHashMap<UUID, TapeDriveCache>();
    private static final Set<UUID> pinnedTapes = ConcurrentHashMap.newKeySet();
    private static ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
    private static final int FLUSH_THRESHOLD = (Integer)serverConfig.TAPE_DB_FLUSH_THRESHOLD.get();
    private static final long FLUSH_INTERVAL = ((Integer)serverConfig.TAPE_DB_FLUSH_INTERVAL.get()).intValue();
    private static final long EVICTION_THRESHOLD = ((Integer)serverConfig.TAPE_DB_RAM_EVICT_TIMEOUT.get()).intValue();
    private static final int PAGE_SIZE = 10000;
    private static final double BYTE_COST_MULTIPLIER = 0.75;
    static boolean debug_log = (Boolean)serverConfig.TAPE_DB_DEBUG_LOG.get();

    public static void init() {
        if (executor.isShutdown() || executor.isTerminated()) {
            executor = Executors.newSingleThreadScheduledExecutor();
        }
        Runtime.getRuntime().addShutdownHook(new Thread(TapeDBManager::flushAll));
        executor.scheduleAtFixedRate(TapeDBManager::flushAndEvict, FLUSH_INTERVAL, FLUSH_INTERVAL, TimeUnit.MILLISECONDS);
        TapeDBManager.log("TapeDBManager initialized.", new Object[0]);
    }

    public static TapeDriveCache getCache(UUID diskId) {
        return activeCaches.get(diskId);
    }

    public static TapeDriveCache getCacheSafe(UUID diskId) {
        return activeCaches.get(diskId);
    }

    public static TapeDriveCache getOrLoadForRead(UUID diskId) {
        return activeCaches.computeIfAbsent(diskId, TapeDBManager::loadFromDisk);
    }

    public static CompletableFuture<TapeDriveCache> loadFromDiskAsync(UUID diskId) {
        if (executor.isShutdown() || executor.isTerminated()) {
            TapeDBManager.init();
        }
        return CompletableFuture.supplyAsync(() -> {
            TapeDriveCache cache = TapeDBManager.loadFromDisk(diskId);
            activeCaches.put(diskId, cache);
            TapeDBManager.notifyAE2StorageChanged(diskId);
            return cache;
        }, executor);
    }

    private static void notifyAE2StorageChanged(UUID diskId) {
    }

    public static long getItemCount(UUID diskId, byte[] itemBytes) {
        TapeDriveCache cache = TapeDBManager.getOrLoadForRead(diskId);
        cache.lastAccessed = System.currentTimeMillis();
        TapeKey key = new TapeKey(itemBytes);
        long committed = cache.entries.getOrDefault(key, StoredEntry.EMPTY).count();
        long delta = cache.deltaBuffer.getOrDefault(key, 0L);
        return Math.max(0L, committed + delta);
    }

    public static int getTypeCount(UUID diskId) {
        TapeDriveCache cache = TapeDBManager.getOrLoadForRead(diskId);
        cache.lastAccessed = System.currentTimeMillis();
        HashSet keys = new HashSet();
        keys.addAll(cache.entries.keySet());
        keys.addAll(cache.deltaBuffer.keySet());
        int count = 0;
        for (TapeKey key : keys) {
            long delta;
            long base = cache.entries.getOrDefault(key, StoredEntry.EMPTY).count();
            if (base + (delta = cache.deltaBuffer.getOrDefault(key, 0L).longValue()) <= 0L) continue;
            ++count;
        }
        return count;
    }

    public static long getTotalStoredBytes(UUID diskId) {
        long count;
        TapeDriveCache cache = TapeDBManager.getCache(diskId);
        if (cache == null) {
            return 0L;
        }
        long total = 0L;
        for (Map.Entry<TapeKey, StoredEntry> entry : cache.entries.entrySet()) {
            count = entry.getValue().count();
            total += Math.round((double)((long)entry.getKey().itemBytes().length * count) * 0.75);
        }
        for (Map.Entry<TapeKey, Object> entry : cache.deltaBuffer.entrySet()) {
            count = (Long)entry.getValue();
            total += Math.round((double)((long)entry.getKey().itemBytes().length * count) * 0.75);
        }
        return Math.max(0L, total);
    }

    public static long getByteLimit(UUID diskId) {
        return ((Integer)serverConfig.TAPE_DISK_BYTE_LIMIT.get()).intValue();
    }

    public static void releaseFromRAM(UUID id) {
        TapeDriveCache cache = activeCaches.remove(id);
        if (cache != null) {
            TapeDBManager.flushAndSave(id, cache);
            TapeDBManager.log("Released and saved tape {} from RAM", id);
        }
    }

    public static Set<UUID> getActiveTapeIds() {
        return new HashSet<UUID>(activeCaches.keySet());
    }

    public static void saveItem(UUID diskId, byte[] itemBytes, AEItemKey key, long delta) {
        long estimated;
        TapeDriveCache cache = TapeDBManager.getCache(diskId);
        if (cache == null) {
            return;
        }
        TapeKey tapeKey = new TapeKey(itemBytes);
        cache.lastAccessed = System.currentTimeMillis();
        cache.deltaBuffer.merge(tapeKey, delta, Long::sum);
        long newItemBytes = (long)itemBytes.length * Math.abs(delta);
        if (delta > 0L && (estimated = cache.totalBytes + newItemBytes) > TapeDBManager.getByteLimit(diskId)) {
            TapeDBManager.log("saveItem rejected for disk %s due to byte limit (%d > %d)", diskId, estimated, TapeDBManager.getByteLimit(diskId));
            return;
        }
        long base = cache.entries.getOrDefault(tapeKey, StoredEntry.EMPTY).count();
        long total = base + cache.deltaBuffer.getOrDefault(tapeKey, 0L);
        if (total <= 0L) {
            cache.entries.remove(tapeKey);
            cache.deltaBuffer.remove(tapeKey);
        }
        File wal = TapeDBManager.getWalFile(diskId);
        try (DataOutputStream dos = new DataOutputStream(new FileOutputStream(wal, true));){
            dos.writeInt(itemBytes.length);
            dos.write(itemBytes);
            dos.writeLong(delta);
            dos.writeLong(TapeDBManager.checksum(itemBytes, delta));
            cache.totalBytes += (long)itemBytes.length * Math.max(delta, 0L);
        }
        catch (IOException e) {
            LOGGER.error("Failed to write WAL for disk {}: {}", (Object)diskId, (Object)e.getMessage());
        }
        if (cache.deltaBuffer.size() >= FLUSH_THRESHOLD) {
            TapeDBManager.flush(diskId, cache);
        }
    }

    public static void flushAll() {
        for (Map.Entry<UUID, TapeDriveCache> entry : activeCaches.entrySet()) {
            TapeDBManager.flush(entry.getKey(), entry.getValue());
        }
        TapeDBManager.log("flushAll complete.", new Object[0]);
    }

    public static void flushAndEvict() {
        long now = System.currentTimeMillis();
        ArrayList<UUID> toEvict = new ArrayList<UUID>();
        for (Map.Entry<UUID, TapeDriveCache> entry : activeCaches.entrySet()) {
            UUID diskId = entry.getKey();
            TapeDriveCache cache = entry.getValue();
            TapeDBManager.flushAndSave(diskId, cache);
            if (TapeDBManager.isPinned(diskId) || now - cache.lastAccessed <= EVICTION_THRESHOLD) continue;
            toEvict.add(diskId);
        }
        for (UUID id : toEvict) {
            activeCaches.remove(id);
            TapeDBManager.log("Evicted tape %s from RAM due to inactivity", id);
        }
    }

    private static void flushAndSave(UUID diskId, TapeDriveCache cache) {
        for (Map.Entry<TapeKey, Long> entry : cache.deltaBuffer.entrySet()) {
            ItemStack stack;
            TapeKey key = entry.getKey();
            long delta = entry.getValue();
            StoredEntry current = cache.entries.getOrDefault(key, new StoredEntry(0L, null));
            long updated = current.count() + delta;
            if (updated <= 0L) {
                cache.entries.remove(key);
                continue;
            }
            AEItemKey aeKey = current.aeKey();
            if (aeKey == null && !(stack = TapeDiskItem.deserializeItemStackFromBytes(key.itemBytes())).isEmpty()) {
                aeKey = AEItemKey.of((ItemStack)stack);
            }
            cache.entries.put(key, new StoredEntry(updated, aeKey));
        }
        cache.deltaBuffer.clear();
        cache.totalBytes = cache.entries.entrySet().stream().mapToLong(e -> (long)((TapeKey)e.getKey()).itemBytes().length * ((StoredEntry)e.getValue()).count()).sum();
        try (DataOutputStream dos = new DataOutputStream(new FileOutputStream(TapeDBManager.getDiskFile(diskId)));){
            for (Map.Entry<TapeKey, StoredEntry> entry : cache.entries.entrySet()) {
                byte[] data = entry.getKey().itemBytes();
                dos.writeInt(data.length);
                dos.write(data);
                dos.writeLong(entry.getValue().count());
            }
        }
        catch (IOException e2) {
            LOGGER.warn("Flush/save failed for disk {}: {}", (Object)diskId, (Object)e2.getMessage());
        }
        try {
            FileOutputStream fos = new FileOutputStream(TapeDBManager.getWalFile(diskId));
            fos.close();
        }
        catch (IOException e3) {
            LOGGER.warn("Failed to clear WAL for disk {}: {}", (Object)diskId, (Object)e3.getMessage());
        }
    }

    public static void shutdown() {
        try {
            TapeDBManager.flushAll();
            executor.shutdown();
            executor.awaitTermination(5L, TimeUnit.SECONDS);
        }
        catch (Exception e) {
            executor.shutdownNow();
        }
        activeCaches.clear();
    }

    private static TapeDriveCache loadFromDisk(UUID diskId) {
        File baseFile = TapeDBManager.getDiskFile(diskId);
        File walFile = TapeDBManager.getWalFile(diskId);
        TapeDriveCache cache = new TapeDriveCache();
        if (baseFile.exists()) {
            ArrayList backupEntries = new ArrayList();
            boolean hadInvalidItems = false;
            try {
                DataInputStream dis = new DataInputStream(new FileInputStream(baseFile));
                try {
                    while (true) {
                        int len = dis.readInt();
                        byte[] data = new byte[len];
                        dis.readFully(data);
                        long count = dis.readLong();
                        ItemStack stack = TapeDiskItem.deserializeItemStackFromBytes(data);
                        if (!stack.isEmpty()) {
                            AEItemKey aeKey = AEItemKey.of((ItemStack)stack);
                            cache.entries.put(new TapeKey(data), new StoredEntry(count, aeKey));
                            continue;
                        }
                        hadInvalidItems = true;
                        LinkedHashMap<String, Object> entry = new LinkedHashMap<String, Object>();
                        entry.put("count", count);
                        entry.put("rawBytes", Base64.getEncoder().encodeToString(data));
                        backupEntries.add(entry);
                    }
                }
                catch (Throwable len) {
                    try {
                        dis.close();
                    }
                    catch (Throwable data) {
                        len.addSuppressed(data);
                    }
                    throw len;
                }
            }
            catch (EOFException dis) {
            }
            catch (IOException e2) {
                LOGGER.warn("Failed reading DB for disk {}: {}", (Object)diskId, (Object)e2.getMessage());
            }
            if (hadInvalidItems) {
                File out = TapeDBManager.getExportFolder().resolve(String.valueOf(diskId) + "_bak.json").toFile();
                try (PrintWriter writer = new PrintWriter(out);){
                    writer.println("[");
                    for (int i = 0; i < backupEntries.size(); ++i) {
                        Map entry = (Map)backupEntries.get(i);
                        writer.println("  {");
                        Iterator it = entry.entrySet().iterator();
                        while (it.hasNext()) {
                            Map.Entry e3 = it.next();
                            writer.print("    \"" + (String)e3.getKey() + "\": \"" + String.valueOf(e3.getValue()) + "\"");
                            if (it.hasNext()) {
                                writer.println(",");
                                continue;
                            }
                            writer.println();
                        }
                        writer.print(i == backupEntries.size() - 1 ? "  }\n" : "  },\n");
                    }
                    writer.println("]");
                    LOGGER.warn("\u00a7e[EnderDrives] Backup JSON created for tape {} due to unreadable entries ({} skipped).", (Object)diskId, (Object)backupEntries.size());
                }
                catch (IOException e4) {
                    LOGGER.error("Failed to write backup JSON for tape {}: {}", (Object)diskId, (Object)e4.getMessage());
                }
            }
        }
        if (walFile.exists()) {
            try {
                DataInputStream dis = new DataInputStream(new FileInputStream(walFile));
                try {
                    while (true) {
                        int len = dis.readInt();
                        byte[] data = new byte[len];
                        dis.readFully(data);
                        long delta = dis.readLong();
                        long checksum = dis.readLong();
                        if (checksum != TapeDBManager.checksum(data, delta)) continue;
                        TapeKey key = new TapeKey(data);
                        AEItemKey aeKey = AEItemKey.of((ItemStack)TapeDiskItem.deserializeItemStackFromBytes(data));
                        long existing = cache.entries.getOrDefault(key, new StoredEntry(0L, aeKey)).count();
                        long updated = existing + delta;
                        if (updated <= 0L) {
                            cache.entries.remove(key);
                            continue;
                        }
                        cache.entries.put(key, new StoredEntry(updated, aeKey));
                    }
                }
                catch (Throwable throwable) {
                    try {
                        dis.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
            }
            catch (EOFException dis) {
            }
            catch (IOException e5) {
                LOGGER.warn("Failed WAL replay for disk {}: {}", (Object)diskId, (Object)e5.getMessage());
            }
            walFile.delete();
        }
        cache.lastAccessed = System.currentTimeMillis();
        cache.totalBytes = cache.entries.entrySet().stream().mapToLong(e -> (long)((TapeKey)e.getKey()).itemBytes().length * ((StoredEntry)e.getValue()).count()).sum();
        return cache;
    }

    private static void flush(UUID diskId, TapeDriveCache cache) {
        if (cache.deltaBuffer.isEmpty()) {
            return;
        }
        for (Map.Entry<TapeKey, Long> entry : cache.deltaBuffer.entrySet()) {
            ItemStack stack;
            TapeKey key = entry.getKey();
            long delta = entry.getValue();
            StoredEntry current = cache.entries.getOrDefault(key, new StoredEntry(0L, null));
            long updated = current.count() + delta;
            if (updated <= 0L) {
                cache.entries.remove(key);
                continue;
            }
            AEItemKey aeKey = current.aeKey();
            if (aeKey == null && !(stack = TapeDiskItem.deserializeItemStackFromBytes(key.itemBytes())).isEmpty()) {
                aeKey = AEItemKey.of((ItemStack)stack);
            }
            cache.entries.put(key, new StoredEntry(updated, aeKey));
        }
        cache.deltaBuffer.clear();
        try (DataOutputStream dos = new DataOutputStream(new FileOutputStream(TapeDBManager.getDiskFile(diskId)));){
            for (Map.Entry<TapeKey, StoredEntry> entry : cache.entries.entrySet()) {
                byte[] data = entry.getKey().itemBytes();
                dos.writeInt(data.length);
                dos.write(data);
                dos.writeLong(entry.getValue().count());
            }
        }
        catch (IOException e) {
            LOGGER.warn("Flush failed for disk {}: {}", (Object)diskId, (Object)e.getMessage());
        }
        try {
            FileOutputStream fos = new FileOutputStream(TapeDBManager.getWalFile(diskId));
            fos.close();
        }
        catch (IOException e) {
            LOGGER.warn("Failed to clear WAL for disk {}: {}", (Object)diskId, (Object)e.getMessage());
        }
    }

    private static long checksum(byte[] data, long delta) {
        CRC32 crc = new CRC32();
        crc.update(data);
        crc.update(TapeDBManager.longToBytes(delta));
        return crc.getValue();
    }

    private static byte[] longToBytes(long val) {
        return new byte[]{(byte)(val >>> 56), (byte)(val >>> 48), (byte)(val >>> 40), (byte)(val >>> 32), (byte)(val >>> 24), (byte)(val >>> 16), (byte)(val >>> 8), (byte)val};
    }

    public static File getDiskFile(UUID id) {
        return TapeDBManager.getFolder().resolve(String.valueOf(id) + ".bin").toFile();
    }

    private static File getWalFile(UUID id) {
        return TapeDBManager.getFolder().resolve(String.valueOf(id) + ".wal").toFile();
    }

    private static Path getFolder() {
        MinecraftServer server = ServerLifecycleHooks.getCurrentServer();
        Path path = server.getWorldPath(LevelResource.ROOT).resolve("data").resolve("enderdrives").resolve("TapeDrives");
        File folder = path.toFile();
        if (!folder.exists()) {
            folder.mkdirs();
        }
        return path;
    }

    private static Path getExportFolder() {
        MinecraftServer server = ServerLifecycleHooks.getCurrentServer();
        Path path = server.getWorldPath(LevelResource.ROOT).resolve("data").resolve("enderdrives").resolve("TapeDrives").resolve("export");
        File folder = path.toFile();
        if (!folder.exists()) {
            folder.mkdirs();
        }
        return path;
    }

    public static boolean exportToJson(UUID tapeId) {
        File dbFile = TapeDBManager.getDiskFile(tapeId);
        if (!dbFile.exists()) {
            return false;
        }
        ArrayList entries = new ArrayList();
        try {
            DataInputStream dis = new DataInputStream(new FileInputStream(dbFile));
            try {
                while (true) {
                    int len = dis.readInt();
                    byte[] data = new byte[len];
                    dis.readFully(data);
                    long count = dis.readLong();
                    ItemStack stack = TapeDiskItem.deserializeItemStackFromBytes(data);
                    if (stack.isEmpty()) continue;
                    LinkedHashMap<String, Object> jsonEntry = new LinkedHashMap<String, Object>();
                    jsonEntry.put("count", count);
                    jsonEntry.put("item", stack.getItem().toString());
                    jsonEntry.put("displayName", stack.getDisplayName().getString());
                    jsonEntry.put("nbt", stack.save((HolderLookup.Provider)ServerLifecycleHooks.getCurrentServer().registryAccess()).toString());
                    entries.add(jsonEntry);
                }
            }
            catch (Throwable len) {
                try {
                    dis.close();
                }
                catch (Throwable data) {
                    len.addSuppressed(data);
                }
                throw len;
            }
        }
        catch (EOFException dis) {
        }
        catch (IOException e) {
            LOGGER.error("Failed to export tape {} to JSON: {}", (Object)tapeId, (Object)e.getMessage());
            return false;
        }
        File out = TapeDBManager.getExportFolder().resolve(String.valueOf(tapeId) + ".json").toFile();
        try (PrintWriter writer = new PrintWriter(out);){
            writer.println("[");
            for (int i = 0; i < entries.size(); ++i) {
                Map entry = (Map)entries.get(i);
                writer.println("  {");
                Iterator it = entry.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry e = it.next();
                    writer.print("    \"" + (String)e.getKey() + "\": \"" + String.valueOf(e.getValue()) + "\"");
                    if (it.hasNext()) {
                        writer.println(",");
                        continue;
                    }
                    writer.println();
                }
                writer.print(i == entries.size() - 1 ? "  }\n" : "  },\n");
            }
            writer.println("]");
        }
        catch (IOException e) {
            LOGGER.error("Failed to write JSON export for tape {}", (Object)tapeId);
            return false;
        }
        return true;
    }

    public static boolean importFromJson(UUID tapeId) {
        File jsonFile = TapeDBManager.getExportFolder().resolve(String.valueOf(tapeId) + ".json").toFile();
        if (!jsonFile.exists()) {
            return false;
        }
        File dbFile = TapeDBManager.getDiskFile(tapeId);
        File walFile = TapeDBManager.getWalFile(tapeId);
        ArrayList<byte[]> serializedItems = new ArrayList<byte[]>();
        ArrayList<Long> counts = new ArrayList<Long>();
        try (BufferedReader reader = new BufferedReader(new FileReader(jsonFile));){
            String line;
            StringBuilder currentNBT = new StringBuilder();
            long currentCount = 0L;
            boolean inObject = false;
            while ((line = reader.readLine()) != null) {
                if ((line = line.trim()).startsWith("{")) {
                    inObject = true;
                    currentNBT.setLength(0);
                    currentCount = 0L;
                    continue;
                }
                if (line.startsWith("\"count\"")) {
                    String raw = line.split(":")[1].replaceAll("[\",]", "").trim();
                    currentCount = Long.parseLong(raw);
                    continue;
                }
                if (line.startsWith("\"nbt\"")) {
                    String nbtRaw = line.substring(line.indexOf(":") + 1).trim();
                    if (nbtRaw.endsWith(",")) {
                        nbtRaw = nbtRaw.substring(0, nbtRaw.length() - 1);
                    }
                    String nbtContent = nbtRaw.replaceFirst("^\"", "").replaceFirst("\"$", "");
                    currentNBT.append(nbtContent);
                    continue;
                }
                if (!line.startsWith("}")) continue;
                inObject = false;
                try {
                    CompoundTag tag = TagParser.parseTag((String)currentNBT.toString());
                    RegistryAccess.Frozen provider = ServerLifecycleHooks.getCurrentServer().registryAccess();
                    ItemStack stack = ItemStack.parse((HolderLookup.Provider)provider, (Tag)tag).orElse(ItemStack.EMPTY);
                    if (stack.isEmpty()) continue;
                    byte[] data = TapeDiskItem.serializeItemStackToBytes(stack);
                    serializedItems.add(data);
                    counts.add(currentCount);
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to import item for tape {}: {}", (Object)tapeId, (Object)e.getMessage());
                }
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed reading export JSON for tape {}: {}", (Object)tapeId, (Object)e.getMessage());
            return false;
        }
        try (DataOutputStream dos = new DataOutputStream(new FileOutputStream(dbFile));){
            for (int i = 0; i < serializedItems.size(); ++i) {
                byte[] data = (byte[])serializedItems.get(i);
                long count = (Long)counts.get(i);
                dos.writeInt(data.length);
                dos.write(data);
                dos.writeLong(count);
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to write new binary DB for tape {}: {}", (Object)tapeId, (Object)e.getMessage());
            return false;
        }
        if (walFile.exists()) {
            walFile.delete();
        }
        TapeDBManager.releaseFromRAM(tapeId);
        LOGGER.info("Successfully imported {} items into tape {} and released from RAM", (Object)serializedItems.size(), (Object)tapeId);
        return true;
    }

    public static List<File> getSortedBinFilesOldestFirst() {
        File folder = TapeDBManager.getFolder().toFile();
        File[] binFiles = folder.listFiles((dir, name) -> name.endsWith(".bin"));
        if (binFiles == null) {
            return List.of();
        }
        List<File> list = Arrays.asList(binFiles);
        list.sort(Comparator.comparingLong(File::lastModified));
        return list;
    }

    public static boolean deleteTape(UUID tapeId) {
        if (activeCaches.containsKey(tapeId)) {
            return false;
        }
        File bin = TapeDBManager.getDiskFile(tapeId);
        File wal = TapeDBManager.getWalFile(tapeId);
        boolean deleted = false;
        if (bin.exists()) {
            deleted |= bin.delete();
        }
        if (wal.exists()) {
            deleted |= wal.delete();
        }
        return deleted;
    }

    public static boolean isPinned(UUID id) {
        return pinnedTapes.contains(id);
    }

    public static void pin(UUID id) {
        pinnedTapes.add(id);
    }

    public static void unpin(UUID id) {
        pinnedTapes.remove(id);
    }

    public static Set<UUID> getPinnedTapes() {
        return Collections.unmodifiableSet(pinnedTapes);
    }

    private static void log(String format, Object ... args) {
        if (debug_log) {
            LOGGER.info(String.format(format, args));
        }
    }

    public static class TapeDriveCache {
        public final ConcurrentHashMap<TapeKey, StoredEntry> entries = new ConcurrentHashMap();
        public final ConcurrentHashMap<TapeKey, Long> deltaBuffer = new ConcurrentHashMap();
        public volatile long lastAccessed = System.currentTimeMillis();
        public long totalBytes = 0L;
    }
}

